<?php
/**
 * NOTICE OF LICENSE
 *
 * This source file is subject to a trade license awarded by
 * Garamo Online S.L.
 *
 * Any use, public communication, reproduction, modification or
 * distribution of this source file without the written consent of
 * Garamo Online S.L. It Is prohibited.
 *
 * @author    ReactionCode <info@reactioncode.com>
 * @copyright Garamo Online S.L 2015-2022.
 * @license   Commercial license
 */

namespace ReactionCode\RCPGTagManager\TrackingAPI\Components\GUA;

use RCPGTagManagerUserTrackingData;
use ReactionCode\RCPGTagManager\DataModel\Connections\ConnectionsModel;
use ReactionCode\RCPGTagManager\DataModel\Order\OrderModel;
use ReactionCode\RCPGTagManager\TheIconic\Tracking\GoogleAnalytics\Analytics;
use ReactionCode\RCPGTagManager\TheIconic\Tracking\GoogleAnalytics\Parameters\EnhancedEcommerce\ProductAction;

class GUAAPIFacade
{
    /**
     * @var Analytics
     */
    private $client;

    /**
     * @var array
     */
    private $configuration;

    /**
     * @param array $configuration
     */
    public function __construct(array $configuration)
    {
        $this->client = new Analytics(true);
        $this->configuration = $configuration;
        $this->initProtocol();
    }

    /**
     *
     */
    private function initProtocol()
    {
        $this->client
            ->setProtocolVersion('1')
            ->setTrackingId($this->configuration['gua']['tracking_id'])
            ->setDataSource('server')
            ->setCacheBuster(time())
        ;
    }

    /**
     * @param OrderModel $order_model
     */
    public function setPurchaseData(OrderModel $order_model)
    {
        $this->client
            ->setCurrencyCode($order_model->getCurrencyCode())
            ->setTransactionId($order_model->getId())
            ->setRevenue($order_model->getTotalAmounts()->getTotalRevenue())
            ->setTax($order_model->getTotalAmounts()->getTotalTax())
            ->setShipping($order_model->getTotalAmounts()->getTotalShipping())
            ->setProductActionToPurchase()
        ;
        $coupons = $order_model->getCoupons();
        if (!empty($coupons)) {
            $this->client->setCouponCode(implode(', ', $coupons));
        }
    }

    /**
     * @param OrderModel $order_model
     */
    public function setRefundData(OrderModel $order_model)
    {
        $this->client
            ->setCurrencyCode($order_model->getCurrencyCode())
            ->setTransactionId($order_model->getId())
            ->setNonInteractionHit(1)
            ->setProductActionToRefund()
        ;
    }

    /**
     * @param ConnectionsModel $connections_model
     */
    public function setTrafficSources(ConnectionsModel $connections_model)
    {
        $document_referrer = $connections_model->referrer->getSourceUrl();
        if (!empty($document_referrer)) {
            $this->client->setDocumentReferrer($document_referrer);
        }
    }

    /**
     * @param ConnectionsModel $source_data
     */
    public function setCampaignData(ConnectionsModel $source_data)
    {
        $landing = $source_data->landing;
        if (!empty($landing->getSourceUrl()) && empty($this->client->getGoogleAdwordsId())) {
            $params = $landing->getParams();
            if (!empty($params['gclid'])) {
                $this->client->setGoogleAdwordsId($params['gclid']);
            }
            if (!empty($params['utm_source'])) {
                $this->client->setCampaignSource($params['utm_source']);
            }
            if (!empty($params['utm_medium'])) {
                $this->client->setCampaignMedium($params['utm_medium']);
            }
            if (!empty($params['utm_campaign'])) {
                $this->client->setCampaignName($params['utm_campaign']);
            }
            if (!empty($params['utm_content'])) {
                $this->client->setCampaignContent($params['utm_content']);
            }
            if (!empty($params['utm_term'])) {
                $this->client->setCampaignKeyword($params['utm_term']);
            }
        }
    }

    /**
     * @param RCPGTagManagerUserTrackingData $user_tracking_data
     */
    public function setUserTrackingData(RCPGTagManagerUserTrackingData $user_tracking_data)
    {
        if (!empty($user_tracking_data->ip)) {
            $this->setIPData($user_tracking_data->ip);
        }

        $tracking_data = $user_tracking_data->getTrackingData();
        $client_id = $tracking_data['ga']['cid'] ?? $user_tracking_data->uuid;
        $this->client->setClientId($client_id);

        if (!empty($tracking_data['ga']['gclid'])) {
            $this->client->setGoogleAdwordsId($tracking_data['ga']['gclid']);
        }
    }

    private function setIPData(string $ip)
    {
        $this->client->setIpOverride($ip);
        if (!empty($this->configuration['gua']['is_anonymize_ip'])) {
            $this->client->setAnonymizeIp(1);
        }
    }

    public function addProduct(GUAAPIProductModel $product_model)
    {
        $product_model->setBasic();
        if (!$this->configuration['gua']['is_data_import'] &&
            $this->client->getProductAction() !== ProductAction::ACTION_REFUND
        ) {
            $product_model->setDetail();
        }
        $this->client->addProduct($product_model->getModel());
    }

    public function setEvent(string $category, string $action, string $label)
    {
        if (!empty($category)) {
            $this->client->setEventCategory($category);
        }
        if (!empty($action)) {
            $this->client->setEventAction($action);
        }
        if (!empty($label)) {
            $this->client->setEventLabel($label);
        }
    }

    public function sendData()
    {
        if ($this->isEvent()) {
            $this->client->sendEvent();
        } else {
            $this->client->sendPageview();
        }
    }

    private function isEvent(): bool
    {
        return $this->client->getEventCategory() ||
            $this->client->getEventAction() ||
            $this->client->getEventLabel()
        ;
    }
}
