<?php
/**
 * NOTICE OF LICENSE
 *
 * This source file is subject to a trade license awarded by
 * Garamo Online S.L.
 *
 * Any use, public communication, reproduction, modification or
 * distribution of this source file without the written consent of
 * Garamo Online S.L. It Is prohibited.
 *
 * @author    ReactionCode <info@reactioncode.com>
 * @copyright Garamo Online S.L 2015-2022.
 * @license   Commercial license
 */

namespace ReactionCode\RCPGTagManager\PSModule\Hook\DisplayBeforeBodyClosingTag;

use Exception;
use Hook;
use Order;
use PrestaShopDatabaseException;
use PrestaShopException;
use RCPGTagManagerOrderTrackingStatus;
use RCPGTagManagerUserTrackingData;
use ReactionCode\RCPGTagManager\DataModel\Order\OrderModelPresenter;
use ReactionCode\RCPGTagManager\DataModel\User\UserModelPresenter;
use ReactionCode\RCPGTagManager\PSModule\DataModelHandler\DataModelHandlerInterface;
use ReactionCode\RCPGTagManager\PSModule\Hook\AbstractHook;
use ReactionCode\RCPGTagManager\PSModule\OrderHandler\Validator\AbstractOrderValidator;
use ReactionCode\RCPGTagManager\UserTrackingData\UserTrackingDataHandler;
use Tools;
use Validate;

abstract class AbstractDisplayBeforeBodyClosingTag extends AbstractHook
{
    /**
     * @var AbstractOrderValidator
     */
    protected $order_validator;

    /**
     * @var array
     */
    protected $tracking_data = [
        'detail_products_list' => [],
        'order_products_list' => [],
        'order_complete_data' => [],
        'order_context_user_data' => []
    ];

    /**
     * @var DataModelHandlerInterface
     */
    private $data_model_handler;

    /**
     * @throws PrestaShopDatabaseException
     * @throws PrestaShopException
     * @throws Exception
     */
    public function exec(array $params = [])
    {
        $this->data_model_handler = $this->module->getServiceProvider()->getDataModelService();

        if ($this->isPaymentController()) {
            $this->processOrderConfirmation();
        } elseif ($this->isCart() || $this->isCheckout()) {
            $this->processCheckout();
        } else {
            $this->processPage();
        }
    }

    /**
     * @return bool
     */
    protected function isPaymentController(): bool
    {
        $executed_hooks = Hook::$executed_hooks;
        return (
            in_array('displayOrderConfirmation', $executed_hooks, true) ||
            in_array('displayPaymentReturn', $executed_hooks, true)
        );
    }

    /**
     * @throws PrestaShopDatabaseException
     * @throws PrestaShopException
     */
    protected function processOrderConfirmation()
    {
        $order_validator = $this->module->getServiceProvider()->getOrderValidatorService();
        $order = $this->fetchOrder();

        if (Validate::isLoadedObject($order) &&
            $order_validator->isValidOrderComplete($order) &&
            RCPGTagManagerOrderTrackingStatus::isPendingTransaction($order)
        ) {
            $order_presenter = new OrderModelPresenter($this->data_model_handler->buildOrderModel($order));
            $user_presenter = new UserModelPresenter($this->data_model_handler->buildUserModel($order));
            $this->tracking_data['order_context_user_data'] = $user_presenter->getModel();
            $this->tracking_data['order_complete_data'] = $order_presenter->getModel();
            $order_products = $order->getOrderDetailList();

            if (!empty($order_products)) {
                $this->tracking_data['detail_products_list'] = $this
                    ->data_model_handler
                    ->buildDetailProductModelList($order_products)
                    ->getPresentedList()
                ;
                $this->tracking_data['order_products_list'] = $this
                    ->data_model_handler
                    ->buildOrderProductModelList($order_products)
                    ->getPresentedList()
                ;
            }
        }
    }

    /**
     * @return Order|null
     * @throws PrestaShopDatabaseException
     * @throws PrestaShopException
     */
    abstract protected function fetchOrder(): ?Order;

    /**
     * @return bool
     */
    abstract protected function isCart(): bool;

    /**
     * @return bool
     */
    protected function isCheckout(): bool
    {
        $controller_name = Tools::getValue('controller');
        $module_detector_service = $this->module->getServiceProvider()->getModuleDetectorService();
        $checkout_module = $module_detector_service->getCheckoutModule();

        return $controller_name === $checkout_module['controller'];
    }

    /**
     * @throws PrestaShopException
     * @throws PrestaShopDatabaseException
     * @throws Exception
     */
    protected function processCheckout()
    {
        $this->setUserTrackingData();

        $cart_products = $this->getCartProducts();

        if (!empty($cart_products)) {
            $this->tracking_data['detail_products_list'] = $this
                ->data_model_handler
                ->buildDetailProductModelList($cart_products)
                ->getPresentedList()
            ;

            $this->tracking_data['order_products_list'] = $this
                ->data_model_handler
                ->buildOrderProductModelList($cart_products)
                ->getPresentedList()
            ;
        }
    }

    /**
     * @throws PrestaShopException
     * @throws PrestaShopDatabaseException
     * @throws Exception
     */
    protected function setUserTrackingData()
    {
        if (!empty($this->context->customer->id)) {
            $user_tracking_data = new UserTrackingDataHandler();
            RCPGTagManagerUserTrackingData::setUserTrackingData(
                $this->context->customer->id,
                $user_tracking_data->getUserTrackingData()
            );
        }
    }

    /**
     * @return array
     */
    protected function getCartProducts(): array
    {
        return $this->context->cart->getProducts();
    }

    protected function processPage()
    {
        $product_view = $this->getTemplateProduct();
        $products_list = $this->getTemplateProductList();

        if (!empty($product_view)) {
            $this->tracking_data['detail_product_view'] = $this
                ->data_model_handler
                ->buildDetailProductModelList($product_view)
                ->getPresentedList()
            ;
        }

        if (!empty($products_list)) {
            $this->tracking_data['detail_products_list'] = $this
                ->data_model_handler
                ->buildDetailProductModelList($products_list)
                ->getPresentedList()
            ;
        }
    }


    /**
     * @return array
     */
    abstract protected function getTemplateProduct(): array;

    /**
     * @return array
     */
    abstract protected function getTemplateProductList(): array;
}
