<?php
/**
 * NOTICE OF LICENSE
 *
 * This source file is subject to a trade license awarded by
 * Garamo Online S.L.
 *
 * Any use, public communication, reproduction, modification or
 * distribution of this source file without the written consent of
 * Garamo Online S.L. It Is prohibited.
 *
 * @author    ReactionCode <info@reactioncode.com>
 * @copyright Garamo Online S.L 2015-2022.
 * @license   Commercial license
 */

namespace ReactionCode\RCPGTagManager\PSModule\Form\Section;

use Configuration;
use Module;
use ReactionCode\RCPGTagManager\PSModule\Form\Transformer\DataTransformerInterface;
use ReactionCode\RCPGTagManager\PSModule\Form\Validator\ValidatorInterface;
use ReactionCode\RCPGTagManager\PSModule\RCTrackingModuleInterface;
use Tools;

abstract class AbstractFormSection implements FormSectionInterface
{
    /**
     * @var Module & RCTrackingModuleInterface
     */
    protected $module;

    protected $errors = [];

    public function __construct(RCTrackingModuleInterface $module)
    {
        $this->module = $module;
    }

    /**
     * @return array
     */
    public function getForm(): array
    {
        $form = [
            'tab_id' => static::FORM_ID,
            'form' => [
                'legend' => [
                    'title' => $this->module->l('Settings'),
                    'icon' => 'icon-cogs'
                ],
                'submit' => [
                    'title' => $this->module->l('Save')
                ]
            ]
        ];

        $content = $this->getContent();

        if (!empty($content)) {
            $form['form'] = array_merge($form['form'], $content);
        }

        return $form;
    }

    /**
     * @return array
     */
    public function getContent(): array
    {
        return [
            'input' => $this->getInputs()
        ];
    }

    /**
     * @return array
     */
    public function getFormValues(): array
    {
        $values = [];
        $inputs = $this->getInputs();
        foreach ($inputs as $input) {
            $values[$input['name']] = $this->getInputValue($input);
        }
        return $values;
    }

    /**
     * @param int|null $id_shop
     * @return array
     */
    public function getModelValues(int $id_shop = null): array
    {
        $values = [];
        $inputs = $this->getInputs();
        foreach ($inputs as $input) {
            $values[$input['name']] = $this->getConfigModelValue($input, $id_shop);
        }
        return $values;
    }

    /**
     * @return array
     */
    public function getErrors(): array
    {
        return $this->errors;
    }

    /**
     * @return bool
     */
    public function isValid(): bool
    {
        $inputs = $this->getInputs();
        $values = $this->getFormValues();
        foreach ($inputs as $input) {
            $value = $values[$input['name']];
            $this->isValidInput($input, $value);
        }
        return !count($this->errors);
    }

    protected function isValidInput(array $input, $value)
    {
        if (!empty($input['required']) && empty($value)) {
            $this->errors[] = $this->module->l(sprintf('"%s" is required', $input['label']));
        }

        if ($this->isTransformer($input)) {
            /** @var DataTransformerInterface $transformer */
            $transformer = $input['transform'];
            $value = $transformer->reverseTransform($value);
        }

        if ($this->isValidator($input) && !empty($value)) {
            /** @var ValidatorInterface $validator */
            $validator = $input['validate'];
            $this->validateValue($validator, $value, $input['label']);
        }
    }

    public function setValues()
    {
        $inputs = $this->getInputs();
        $values = $this->getFormValues();
        foreach ($inputs as $input) {
            $config_value = $values[$input['name']];

            if ($this->isTransformer($input)) {
                /** @var DataTransformerInterface $transformer */
                $transformer = $input['transform'];
                $config_value = $transformer->reverseTransform($config_value);
            }

            if (!empty($input['is_array_model'])) {
                $config_value = json_encode($config_value);
            }

            Configuration::updateValue($this->getConfigurationKey($input['name']), $config_value);
        }
    }

    /**
     * @return bool
     */
    public function setDefaultValues(): bool
    {
        $result = true;
        $inputs = $this->getInputs();
        foreach ($inputs as $input) {
            $key = $this->getConfigurationKey($input['name']);
            if (!Configuration::get($key) && isset($input['default_value'])) {
                $default_value = $input['default_value'];

                if (!empty($input['is_array_model'])) {
                    $default_value = json_encode($default_value);
                }

                $result = $result && Configuration::updateGlobalValue($key, $default_value);
            }
        }
        return $result;
    }

    private function getInputValue(array $input)
    {
        return Tools::getValue($input['name'], $this->getConfigValue($input));
    }

    /**
     * @param array $input
     * @param int|null $id_shop
     * @return false|mixed|string
     */
    private function getConfigValue(array $input, int $id_shop = null)
    {
        $config_value = $this->getConfigModelValue($input, $id_shop);

        if ($this->isTransformer($input)) {
            /** @var DataTransformerInterface $transformer */
            $transformer = $input['transform'];
            $config_value = $transformer->transform($config_value);
        }
        return $config_value;
    }

    private function getConfigModelValue(array $input, int $id_shop = null)
    {
        $config_value = Configuration::get($this->getConfigurationKey($input['name']), null, null, $id_shop);

        if (!empty($input['is_array_model'])) {
            $config_value = json_decode($config_value, true);
        }

        return $config_value;
    }

    /**
     * @param string $key
     * @return string
     */
    private function getConfigurationKey(string $key): string
    {
        return Tools::strtoupper($this->module->name.'_'.$key);
    }

    private function isTransformer(array $input): bool
    {
        return !empty($input['transform']) && $input['transform'] instanceof  DataTransformerInterface;
    }

    private function isValidator($input): bool
    {
        return !empty($input['validate']) && $input['validate'] instanceof ValidatorInterface;
    }

    private function validateValue(ValidatorInterface $validator, $value, $input_label)
    {
        $validate_values = [];

        if (!is_array($value)) {
            $validate_values[] = $value;
        } else {
            $validate_values = $value;
        }
        foreach ($validate_values as $value) {
            $is_valid = $validator->isValid($value);
            if (!$is_valid) {
                $tab = $this->getTab();
                $this->errors[] = $this->module->l(
                    sprintf('"%s" from "%s" tab, has invalid value: %s', $input_label, $tab['name'], $value)
                );
            }
        }
    }
}
