<?php
/**
 * NOTICE OF LICENSE
 *
 * This source file is subject to a trade license awarded by
 * Garamo Online S.L.
 *
 * Any use, public communication, reproduction, modification or
 * distribution of this source file without the written consent of
 * Garamo Online S.L. It Is prohibited.
 *
 * @author    ReactionCode <info@reactioncode.com>
 * @copyright Garamo Online S.L 2015-2022.
 * @license   Commercial license
 */

namespace ReactionCode\RCPGTagManager\PSModule\DataModelHandler\ModelBuilder\User;

use Context;
use Customer;
use DateTime;
use Exception;
use Gender;
use Group;
use Order;
use PrestaShopDatabaseException;
use PrestaShopException;
use ReactionCode\RCPGTagManager\DataModel\User\GroupModel;
use ReactionCode\RCPGTagManager\DataModel\User\UserModel;
use ReactionCode\RCPGTagManager\PSModule\Cookie\CookieConstant;
use Tools;

abstract class AbstractUserModelBuilder implements UserModelBuilderInterface
{
    /**
     * @var Customer
     */
    protected $customer;

    /**
     * @var Order
     */
    protected $order;

    /**
     * @var UserModel
     */
    protected $model;

    /**
     * @var Context
     */
    protected $context;

    public function __construct()
    {
        $this->context = Context::getContext();
    }

    /**
     * @param Customer $customer
     * @param Order|null $order
     * @throws PrestaShopDatabaseException
     * @throws PrestaShopException
     */
    public function initModel(Customer $customer, Order $order = null)
    {
        $this->model = new UserModel();
        if (!empty($customer->id)) {
            $this->customer = $customer;
            $this->order = $order;
            $this->setCustomerData();
        }
    }

    /**
     * @throws PrestaShopDatabaseException
     * @throws PrestaShopException
     * @throws Exception
     */
    protected function setCustomerData()
    {
        $this->model->setIdUser((int)$this->customer->id);
        $this->model->setIdGender($this->getGenderType());
        $this->model->setIdDefaultGroup((int)$this->customer->id_default_group);
        $this->model->setBirthday($this->getBirthday());
        $this->model->setFirstName((string)$this->customer->firstname);
        $this->model->setLastName((string)$this->customer->lastname);
        $this->model->setEmail((string)$this->customer->email);
        $this->model->setIsGuest((bool)$this->customer->is_guest);
        $this->model->setIsNewsletter((bool)$this->customer->newsletter);
        $this->model->setIsOptIn((bool)$this->customer->optin);

        $this->setGroupModel();
        $this->setAddressModel();
        $this->setStatsModel();

        if (!Tools::getIsset('submitLogin') && !Tools::getIsset('submitCreate')) {
            $this->setIsNewAccount();
            $this->setIsNewLogin();
        }
    }

    /**
     * @return int
     */
    protected function getGenderType(): int
    {
        $gender = new Gender($this->customer->id_gender);
        return (int)$gender->type;
    }

    /**
     * @return void
     */
    public function setIsNewAccount()
    {
        if (isset($this->context->cookie->{CookieConstant::NEW_ACCOUNT})) {
            $this->model->setIsNewAccount(true);
            $this->context->cookie->__unset(CookieConstant::NEW_ACCOUNT);
        }
    }

    /**
     * @return void
     */
    public function setIsNewLogin()
    {
        if (isset($this->context->cookie->{CookieConstant::NEW_LOGIN})) {
            $this->model->setIsNewLogin(true);
            $this->context->cookie->__unset(CookieConstant::NEW_LOGIN);
        }
    }

    /**
     * @return void
     * @throws PrestaShopDatabaseException
     * @throws PrestaShopException
     */
    protected function setGroupModel()
    {
        $group_model_list = $this->model->getGroupModelList();

        $groups = $this->customer->getGroups();

        foreach ($groups as $id_group) {
            $group = new Group($id_group, $this->customer->id_lang);
            $group_model = new GroupModel();
            $group_model->setId((int)$group->id);
            $group_model->setName((string)$group->name);
            $group_model_list->addGroup($group_model);
        }
    }

    /**
     * @return void
     */
    abstract protected function setAddressModel();

    /**
     * @return int
     */
    protected function getIdAddress(): int
    {
        return (int)(
            ($this->order instanceof Order) ?
                $this->order->id_address_delivery :
                $this->context->cart->id_address_delivery
            )
        ;
    }

    /**
     * @return void
     */
    protected function setStatsModel()
    {
        $orders = Order::getCustomerOrders($this->customer->id);
        $validOrders = array_filter($orders, [$this, 'filterValidOrders']);
        $total_spent = array_reduce($validOrders, [$this, 'calcTotalSpent'], 0);

        $stats_model = $this->model->getStatsModel();
        $stats_model->setOrdersPlaced(count($validOrders));
        $stats_model->setTotalSpent($total_spent);
    }

    /**
     * @param array $order
     * @return bool
     */
    private function filterValidOrders(array $order): bool
    {
        return (bool)$order['valid'];
    }

    /**
     * @param $total
     * @param $order
     * @return mixed
     */
    private function calcTotalSpent($total, $order)
    {
        return $total + $order['total_paid'];
    }

    /**
     * @return int
     * @throws Exception
     */
    private function getBirthday(): int
    {
        $date = new DateTime($this->customer->birthday);
        $birthday = (int)$date->format('Ymd');
        return ($birthday > 0) ? $birthday : 0;
    }

    /**
     * @return UserModel
     */
    public function getModel(): UserModel
    {
        return $this->model;
    }
}
