<?php
/**
 * NOTICE OF LICENSE
 *
 * This source file is subject to a trade license awarded by
 * Garamo Online S.L.
 *
 * Any use, public communication, reproduction, modification or
 * distribution of this source file without the written consent of
 * Garamo Online S.L. It Is prohibited.
 *
 * @author    ReactionCode <info@reactioncode.com>
 * @copyright Garamo Online S.L 2015-2022.
 * @license   Commercial license
 */

namespace ReactionCode\RCPGTagManager\PSModule\DataModelHandler\ModelBuilder\Product;

use Context;
use Manufacturer;
use Product;
use ReactionCode\RCPGTagManager\DataModel\Product\ProductModel;
use ReactionCode\RCPGTagManager\PSModule\DataModelHandler\ModelBuilder\Category\CategoryModelBuilderInterface;
use StockAvailable;
use Tools;

abstract class AbstractProductModelBuilder implements ProductModelBuilderInterface
{
    const PRICE_PRECISION = 2;

    /**
     * @var Product
     */
    private $product;

    /**
     * @var ProductModel
     */
    private $model;

    /**
     * @param int $id_product
     * @param int $id_product_attribute
     * @param int $id_lang
     */
    public function initModel(int $id_product, int $id_product_attribute = 0, int $id_lang = 0)
    {
        $id_lang = $id_lang ?: Context::getContext()->language->id;
        $this->product = new Product($id_product, false, $id_lang);
        $this->model = new ProductModel($id_product, $id_product_attribute, $id_lang);
    }

    /**
     * @param CategoryModelBuilderInterface $category_builder
     */
    public function setDetailData(CategoryModelBuilderInterface $category_builder)
    {
        $category_builder->initModel((int)$this->product->id_category_default, $this->model->getIdLang());
        $this->model->getCategoryModel()->setIdCategory($category_builder->getModel()->getIdCategory());
        $this->model->getCategoryModel()->setPath($category_builder->getModel()->getPath());

        $this->model->setName((string)$this->product->name);
        $this->model->setAttributes($this->getVariantData());
        $this->model->setIdManufacturer((int)$this->product->id_manufacturer);
        $this->model->setManufacturerName($this->getManufacturerName());
        $this->model->setEan((string)$this->product->ean13);
        $this->model->setReference((string)$this->product->reference);
        $this->model->setUpc((string)$this->product->upc);
        $this->model->setCondition((string)$this->product->condition);
        $this->model->setIsAvailable($this->getIsAvailable());
        $this->model->setPrice($this->getPrice());
        $this->model->setPriceTaxExcl($this->getPrice(false));
        $this->model->setPriceOld($this->getPrice(true, false));
        $this->model->setPriceOldTaxExcl($this->getPrice(false, false));
    }

    /**
     * @param float $price
     * @param float $price_tax_excl
     * @param int $quantity
     */
    public function setOderData(float $price, float $price_tax_excl, int $quantity)
    {
        $this->model->setPrice(Tools::ps_round($price, 2));
        $this->model->setPriceTaxExcl(Tools::ps_round($price_tax_excl, 2));
        $this->model->setQuantity($quantity);
    }

    /**
     * @return array
     */
    protected function getVariantData(): array
    {
        $product_attributes = [];

        if (!empty($this->model->getIdAttribute())) {
            $attributes_params = Product::getAttributesParams(
                $this->model->getIdProduct(),
                $this->model->getIdAttribute()
            );
            $product_attributes = array_map([$this, 'variantMap'], $attributes_params);
        }
        return $product_attributes;
    }

    /**
     * @param $attribute
     * @return array
     */
    protected function variantMap($attribute): array
    {
        return [$attribute['group'], $attribute['name']];
    }

    /**
     * @return string
     */
    protected function getManufacturerName(): string
    {
        return Manufacturer::getNameById($this->model->getIdManufacturer());
    }

    /**
     * @param bool $use_tax
     * @param bool $use_reduction
     * @return float
     */
    protected function getPrice(bool $use_tax = true, bool $use_reduction = true): float
    {
        return (float)Product::getPriceStatic(
            $this->model->getIdProduct(),
            $use_tax,
            $this->model->getIdAttribute(),
            self::PRICE_PRECISION,
            null,
            false,
            $use_reduction
        );
    }

    /**
     * @return bool
     */
    protected function getIsAvailable(): bool
    {
        return (bool)StockAvailable::getQuantityAvailableByProduct(
            $this->model->getIdProduct(),
            $this->model->getIdAttribute()
        );
    }

    /**
     * @return ProductModel
     */
    public function getModel(): ProductModel
    {
        return $this->model;
    }
}
