<?php
/**
 * NOTICE OF LICENSE
 *
 * This source file is subject to a trade license awarded by
 * Garamo Online S.L.
 *
 * Any use, public communication, reproduction, modification or
 * distribution of this source file without the written consent of
 * Garamo Online S.L. It Is prohibited.
 *
 * @author    ReactionCode <info@reactioncode.com>
 * @copyright Garamo Online S.L 2015-2022.
 * @license   Commercial license
 */

namespace ReactionCode\RCPGTagManager\PSModule\DataModelHandler\ModelBuilder\Order;

use Carrier;
use Currency;
use Order;
use ReactionCode\RCPGTagManager\DataModel\Order\OrderModel;
use Tools;

abstract class AbstractOrderModelBuilder implements OrderModelBuilderInterface
{
    /**
     * @var Order
     */
    private $order;

    /**
     * @var OrderModel
     */
    private $model;

    public function initModel(Order $order)
    {
        $this->order = $order;
        $this->model = new OrderModel();
        $this->setOrderData();
    }

    protected function setOrderData()
    {
        $this->model->setId((int)$this->order->id);
        $this->model->setIdCustomer((int)$this->order->id_customer);
        $this->model->setIdLang((int)$this->order->id_lang);
        $this->model->setIdShop((int)$this->order->id_shop);
        $this->model->setCoupons($this->getCoupons());
        $this->model->setCurrencyCode((string)Currency::getCurrencyInstance($this->order->id_currency)->iso_code);
        $this->model->setDateAdd((string)$this->order->date_add);
        $this->model->setCarrierName((string)$this->getCarrierName());
        $this->model->setPaymentModule((string)$this->order->module);
        $this->model->setPaymentName((string)$this->order->payment);
        $this->setTotalAmounts();
    }

    /**
     * @return array
     */
    private function getCoupons(): array
    {
        $coupons = [];
        $cart_rules = $this->order->getCartRules();

        if (is_array($cart_rules)) {
            $coupons = array_column($cart_rules, 'name');
        }
        return $coupons;
    }

    /**
     * @return string
     */
    private function getCarrierName(): string
    {
        $carrier = new Carrier($this->order->id_carrier, $this->order->id_lang);
        return $carrier->name;
    }

    private function setTotalAmounts()
    {
        $total_amounts = $this->model->getTotalAmounts();

        $total_amounts->setTotalDiscounts((float)$this->roundValue($this->order->total_discounts));
        $total_amounts->setTotalDiscountsTaxExcl((float)$this->roundValue($this->order->total_discounts_tax_excl));
        $total_amounts->setTotalRevenue((float)$this->roundValue($this->order->total_paid));
        $total_amounts->setTotalRevenueTaxExcl((float)$this->roundValue($this->order->total_paid_tax_excl));
        $total_amounts->setTotalProducts((float)$this->roundValue($this->order->total_products_wt));
        $total_amounts->setTotalProductsTaxExcl((float)$this->roundValue($this->order->total_products));
        $total_amounts->setTotalShipping((float)$this->roundValue($this->order->total_shipping));
        $total_amounts->setTotalShippingTaxExcl((float)$this->roundValue($this->order->total_shipping_tax_excl));
        $total_amounts->setTotalWrapping((float)$this->roundValue($this->order->total_wrapping));
        $total_amounts->setTotalWrappingTaxExcl((float)$this->roundValue($this->order->total_wrapping_tax_excl));
        $total_amounts->setTotalTax((float)$this->calcTotalTax());
    }

    /**
     * @param $value
     * @return float
     */
    private function roundValue($value): float
    {
        return Tools::ps_round($value, 2);
    }

    /**
     * @return float
     */
    private function calcTotalTax(): float
    {
        $total_paid_tax_incl = $this->roundValue($this->order->total_paid_tax_incl);
        $total_paid_tax_excl = $this->roundValue($this->order->total_paid_tax_excl);
        return  $total_paid_tax_incl - $total_paid_tax_excl;
    }

    public function getModel(): OrderModel
    {
        return $this->model;
    }
}
