<?php
/**
 * NOTICE OF LICENSE
 *
 * This source file is subject to a trade license awarded by
 * Garamo Online S.L.
 *
 * Any use, public communication, reproduction, modification or
 * distribution of this source file without the written consent of
 * Garamo Online S.L. It Is prohibited.
 *
 * @author    ReactionCode <info@reactioncode.com>
 * @copyright Garamo Online S.L 2015-2022.
 * @license   Commercial license
 */

namespace ReactionCode\RCPGTagManager\PSModule\DataModelHandler;

use Context;
use Exception;
use Order;
use PrestaShopLogger;
use ReactionCode\RCPGTagManager\DataModel\Category\CategoryModel;
use ReactionCode\RCPGTagManager\DataModel\Order\OrderModel;
use ReactionCode\RCPGTagManager\DataModel\Product\ProductModelList;
use ReactionCode\RCPGTagManager\DataModel\Product\ProductModelListInterface;
use ReactionCode\RCPGTagManager\DataModel\Search\SearchModel;
use ReactionCode\RCPGTagManager\DataModel\User\UserModel;
use ReactionCode\RCPGTagManager\PSModule\Adapter\Product\ProductAdapterInterface;
use ReactionCode\RCPGTagManager\PSModule\DataModelHandler\ModelBuilder\ModelBuilderFactoryInterface;

class DataModelHandler implements DataModelHandlerInterface
{
    /**
     * @var ModelBuilderFactoryInterface
     */
    private $model_builder_factory;
    /**
     * @var ProductAdapterInterface
     */
    private $product_adapter;
    /**
     * @var Context
     */
    private $context;

    /**
     * @param ModelBuilderFactoryInterface $model_builder_factory
     * @param ProductAdapterInterface $product_adapter
     */
    public function __construct(
        ModelBuilderFactoryInterface $model_builder_factory,
        ProductAdapterInterface $product_adapter
    ) {
        $this->model_builder_factory = $model_builder_factory;
        $this->product_adapter = $product_adapter;
        $this->context = Context::getContext();
    }

    /**
     * @param int $id_category
     * @param int $id_lang
     * @return CategoryModel
     */
    public function buildCategoryModel(int $id_category, int $id_lang): CategoryModel
    {
        $builder = $this->model_builder_factory->createCategoryModelBuilder();
        $builder->initModel($id_category, $id_lang);
        return $builder->getModel();
    }

    /**
     * @return SearchModel
     */
    public function buildSearchModel(): SearchModel
    {
        $builder = $this->model_builder_factory->createSearchModelBuilder();
        $builder->initModel();
        return $builder->getModel();
    }

    /**
     * @param array $products
     * @param int $id_lang
     * @return ProductModelListInterface
     * @throws Exception
     */
    public function buildDetailProductModelList(array $products, int $id_lang = 0): ProductModelListInterface
    {
        $model_list = new ProductModelList();
        $product_builder = $this->model_builder_factory->createProductModelBuilder();
        foreach ($products as $product) {
            $product = $this->product_adapter->normalizeProduct($product);

            if (isset($product['id_product'], $product['id_product_attribute'])) {
                $product_builder->initModel(
                    (int)$product['id_product'],
                    (int)$product['id_product_attribute'],
                    (int)$id_lang
                );
                $product_builder->setDetailData($this->model_builder_factory->createCategoryModelBuilder());
                $model_list->addProduct($product_builder->getModel());
            } else {
                PrestaShopLogger::addLog(
                    'product params are not valid, actually: ( '.json_encode($product).' )'
                );
            }
        }
        return $model_list;
    }

    /**
     * @param array $products
     * @param int $id_lang
     * @return ProductModelListInterface
     * @throws Exception
     */
    public function buildOrderProductModelList(array $products, int $id_lang = 0): ProductModelListInterface
    {
        $model_list = new ProductModelList();
        $product_builder = $this->model_builder_factory->createProductModelBuilder();
        foreach ($products as $product) {
            $product = $this->product_adapter->normalizeProduct($product);

            if (isset($product['id_product'], $product['id_product_attribute'])) {
                $product_builder->initModel(
                    (int)$product['id_product'],
                    (int)$product['id_product_attribute'],
                    (int)$id_lang
                );
                $product_builder->setOderData(
                    $product['price'],
                    $product['price_tax_excl'],
                    $this->getProductOrderQuantity($product)
                );
                $model_list->addProduct($product_builder->getModel());
            } else {
                PrestaShopLogger::addLog(
                    'product params are not valid, actually: ( '.json_encode($product).' )'
                );
            }
        }
        return $model_list;
    }

    /**
     * @param array $products
     * @param int $id_lang
     * @return ProductModelListInterface
     * @throws Exception
     */
    public function buildAPIProductModelList(array $products, int $id_lang = 0): ProductModelListInterface
    {
        $model_list = new ProductModelList();
        $product_builder = $this->model_builder_factory->createProductModelBuilder();
        foreach ($products as $product) {
            $product = $this->product_adapter->normalizeProduct($product);

            if (isset($product['id_product'], $product['id_product_attribute'])) {
                $product_builder->initModel(
                    (int)$product['id_product'],
                    (int)$product['id_product_attribute'],
                    (int)$id_lang
                );
                $product_builder->setDetailData($this->model_builder_factory->createCategoryModelBuilder());
                $product_builder->setOderData(
                    $product['price'],
                    $product['price_tax_excl'],
                    $this->getProductOrderQuantity($product)
                );
                $model_list->addProduct($product_builder->getModel());
            } else {
                PrestaShopLogger::addLog(
                    'product params are not valid, actually: ( '.json_encode($product).' )'
                );
            }
        }
        return $model_list;
    }

    /**
     * @param array $product
     * @return int
     */
    protected function getProductOrderQuantity(array $product): int
    {
        $quantity = 0;
        // quantity exists on cart, checkout, order and refund
        if (!empty($product['quantity'])) {
            $quantity = $product['quantity'];
        }

        // quantity refunded exist on refund
        if (!empty($product['quantity_refunded'])) {
            $quantity = $product['quantity_refunded'];
        }

        return (int)$quantity;
    }

    /**
     * @param Order $order
     * @return OrderModel
     */
    public function buildOrderModel(Order $order): OrderModel
    {
        $builder = $this->model_builder_factory->createOrderModelBuilder();
        $builder->initModel($order);
        return $builder->getModel();
    }

    /**
     * @param Order|null $order
     * @return UserModel
     */
    public function buildUserModel(Order $order = null): UserModel
    {
        $builder = $this->model_builder_factory->createUserModelBuilder();
        $builder->initModel($this->context->customer, $order);
        return $builder->getModel();
    }
}
