<?php
/**
 * NOTICE OF LICENSE
 *
 * This source file is subject to a trade license awarded by
 * Garamo Online S.L.
 *
 * Any use, public communication, reproduction, modification or
 * distribution of this source file without the written consent of
 * Garamo Online S.L. It Is prohibited.
 *
 * @author    ReactionCode <info@reactioncode.com>
 * @copyright Garamo Online S.L 2015-2022.
 * @license   Commercial license
 */

namespace ReactionCode\RCPGTagManager\PSModule\Cron;

use Context;
use DateInterval;
use DateTime;
use Db;
use Module;
use Order;
use PrestaShopDatabaseException;
use PrestaShopException;
use RCPGTagManagerOrderTrackingStatus;
use ReactionCode\RCPGTagManager\PSModule\APIHandler\APIHandlerInterface;
use ReactionCode\RCPGTagManager\PSModule\OrderHandler\Validator\OrderValidatorInterface;
use ReactionCode\RCPGTagManager\PSModule\RCTrackingModuleInterface;

class CronHandler
{
    /**
     * @var Module & RCTrackingModuleInterface
     */
    protected $module;

    /**
     * @var APIHandlerInterface
     */
    protected $api_handler;

    /**
     * @var OrderValidatorInterface
     */
    protected $order_validator;

    /**
     * @param RCTrackingModuleInterface $module
     */
    public function __construct(RCTrackingModuleInterface $module)
    {
        $this->module = $module;
        $this->api_handler = $this->module->getServiceProvider()->getAPIService();
        $this->order_validator = $this->module->getServiceProvider()->getOrderValidatorService();
    }

    /**
     * @throws PrestaShopDatabaseException
     * @throws PrestaShopException
     */
    public function recoverMissingTransactions()
    {
        $missing_orders = $this->getMissingOrders();
        foreach ($missing_orders as $missing_order) {
            $order = new Order($missing_order['id_order']);
            if ($this->order_validator->isValidOrderComplete($order)) {
                $this->api_handler->sendOrderComplete($order);
                RCPGTagManagerOrderTrackingStatus::setOrderStatus(
                    $order,
                    RCPGTagManagerOrderTrackingStatus::SERVER_CRON
                );
            }
        }
    }

    /**
     * @return array
     * @throws PrestaShopDatabaseException
     */
    private function getMissingOrders(): array
    {
        $query = $this->getMissingOrdersQuery();
        $result = Db::getInstance()->executeS($query);
        return is_array($result) ? $result : [];
    }

    /**
     * @return string
     */
    private function getMissingOrdersQuery(): string
    {
        $table_name = RCPGTagManagerOrderTrackingStatus::$definition['table'];
        $context = Context::getContext();
        $query_dates = $this->getQueryDates();

        return 'SELECT o.id_order
            FROM ' . _DB_PREFIX_ . 'orders AS o
            LEFT JOIN ' . _DB_PREFIX_ . $table_name . ' AS ots
            ON ots.id_order = o.id_order
            WHERE o.id_shop = "' . (int)$context->shop->id .'"
            AND o.date_add BETWEEN "' . pSQL($query_dates['min']) . '" AND "' . pSQL($query_dates['max']) . '"  
            AND (
                ots.status IN (
                ' . (int)RCPGTagManagerOrderTrackingStatus::WAITING_THANK_YOU_PAGE . ',
                ' . (int)RCPGTagManagerOrderTrackingStatus::ABORTED_INVALID_STATUS . ' 
                ) OR
                ots.status IS NULL
            ) 
            ORDER BY o.date_add DESC'
        ;
    }

    /**
     * @return array
     */
    private function getQueryDates(): array
    {
        $date_time = new DateTime();
        $format = 'Y-m-d H:i:s';
        return [
            'max' => $date_time->sub(new DateInterval('PT10M'))->format($format),
            'min' => $date_time->sub(new DateInterval('P3D'))->format($format)
        ];
    }
}
