<?php
/**
 * NOTICE OF LICENSE
 *
 * This source file is subject to a trade license awarded by
 * Garamo Online S.L.
 *
 * Any use, public communication, reproduction, modification or
 * distribution of this source file without the written consent of
 * Garamo Online S.L. It Is prohibited.
 *
 * @author    ReactionCode <info@reactioncode.com>
 * @copyright Garamo Online S.L 2015-2022.
 * @license   Commercial license
 */

namespace ReactionCode\RCPGTagManager\PSModule\APIHandler\Components\Connections;

use Context;
use DateTime;
use Db;
use Exception;
use Module;
use PrestaShopDatabaseException;
use ReactionCode\RCPGTagManager\DataModel\Connections\ConnectionsModel;
use ReactionCode\RCPGTagManager\DataModel\Order\OrderModel;
use ReactionCode\RCPGTagManager\PSModule\RCTrackingModuleInterface;

class AbstractOrderConnections implements OrderConnectionsInterface
{
    /**
     * @var Module & RCTrackingModuleInterface
     */
    protected $module;

    /**
     * @var Context
     */
    protected $context;

    /**
     * @var array
     */
    protected $connections;

    /**
     * @var array
     */
    protected $api_common_configuration;

    /**
     * @param RCTrackingModuleInterface $module
     */
    public function __construct(RCTrackingModuleInterface $module)
    {
        $this->module = $module;
        $this->context = Context::getContext();
        $this->setAPICommonConfiguration();
    }

    private function setAPICommonConfiguration()
    {
        $configurator_service = $this->module->getServiceProvider()->getConfiguratorService();
        $this->api_common_configuration = $configurator_service->getAPICommonConfiguration();
    }

    /**
     * @param OrderModel $order
     * @return ConnectionsModel
     * @throws PrestaShopDatabaseException
     */
    public function getSourceConnection(OrderModel $order): ConnectionsModel
    {
        $this->connections = $this->getAllConnections($order);
        $filtered_connections = $this->getFilteredConnection();
        return new ConnectionsModel($filtered_connections);
    }

    /**
     * @param OrderModel $order
     * @return array
     * @throws PrestaShopDatabaseException
     * @throws Exception
     */
    private function getAllConnections(OrderModel $order): array
    {
        $query = $this->getConnectionsQuery($order);
        $result = Db::getInstance()->executeS($query);
        return is_array($result) ? $result : [];
    }

    /**
     * @param OrderModel $order
     * @return string
     * @throws Exception
     */
    protected function getConnectionsQuery(OrderModel $order): string
    {
        $campaign_timeout_date = $this->getCampaignTimeoutDate($order);
        return 'SELECT cos.http_referer, cos.request_uri, cos.keywords
            FROM ' . _DB_PREFIX_ . 'orders o
            INNER JOIN ' . _DB_PREFIX_ . 'guest g ON g.id_customer = o.id_customer
            INNER JOIN ' . _DB_PREFIX_ . 'connections co  ON co.id_guest = g.id_guest
            INNER JOIN ' . _DB_PREFIX_ . 'connections_source cos ON cos.id_connections = co.id_connections
            WHERE id_order = ' . (int)($order->getId()) . ' 
            AND co.date_add <= "' . pSQL($order->getDateAdd()) . '" 
            AND co.date_add >= "' . pSQL($campaign_timeout_date) . '"
            ORDER BY cos.date_add DESC'
        ;
    }

    /**
     * @throws Exception
     */
    private function getCampaignTimeoutDate(OrderModel $order): string
    {
        $order_date = $order->getDateAdd();
        $campaign_timeout = $this->api_common_configuration['campaign_timeout'];
        $order_date_time = new DateTime($order_date);
        $order_date_time->modify('-'.$campaign_timeout.' month');
        return $order_date_time->format('Y-m-d h:m:s');
    }

    /**
     * @return array
     */
    private function getFilteredConnection(): array
    {
        $group_match = $this->getGroupMatchExp();
        $pattern_rel = '/((?![.\-])[a-z0-9-]*\.)?(?:' . $group_match . ')$/iA';

        if (!empty($this->connections)) {
            foreach ($this->connections as $connection) {
                $host = parse_url($connection['http_referer'], PHP_URL_HOST);
                if (!preg_match($pattern_rel, $host)) {
                    return $connection;
                }
            }
        }
        return [];
    }

    /**
     * @return string
     */
    private function getGroupMatchExp(): string
    {
        $group_separator = '|';
        $referral_exclusion_list = implode(
            $group_separator,
            $this->api_common_configuration['referral_exclusion_list']
        );

        $search = '.';
        $replace = '\.';
        return str_replace($search, $replace, $referral_exclusion_list);
    }
}
