<?php
/**
 * NOTICE OF LICENSE
 *
 * This source file is subject to a trade license awarded by
 * Garamo Online S.L.
 *
 * Any use, public communication, reproduction, modification or
 * distribution of this source file without the written consent of
 * Garamo Online S.L. It Is prohibited.
 *
 * @author    ReactionCode <info@reactioncode.com>
 * @copyright Garamo Online S.L 2015-2022.
 * @license   Commercial license
 */

namespace ReactionCode\RCPGTagManager\PSModule\APIHandler;

use Module;
use Order;
use PrestaShopDatabaseException;
use PrestaShopException;
use RCPGTagManagerUserTrackingData;
use ReactionCode\RCPGTagManager\DataModel\Connections\ConnectionsModel;
use ReactionCode\RCPGTagManager\DataModel\Order\OrderModel;
use ReactionCode\RCPGTagManager\PSModule\DataModelHandler\DataModelHandlerInterface;
use ReactionCode\RCPGTagManager\PSModule\RCTrackingModuleInterface;
use ReactionCode\RCPGTagManager\TrackingAPI\APIProvider;
use ReactionCode\RCPGTagManager\TrackingAPI\Components\APIComponentInterface;

abstract class AbstractAPIHandler implements APIHandlerInterface
{
    /**
     * @var Module & RCTrackingModuleInterface
     */
    protected $module;

    /**
     * @var APIComponentInterface[]
     */
    private $available_apis;

    /**
     * @var DataModelHandlerInterface
     */
    private $data_model_handler;

    public function __construct(RCTrackingModuleInterface $module)
    {
        $this->module = $module;
        $this->available_apis = $this->getAvailableAPIS();
        $this->data_model_handler = $module->getServiceProvider()->getDataModelService();
    }

    /**
     * @return array
     */
    private function getAvailableAPIS(): array
    {
        $tracking_factory = new APIProvider($this->module);
        return $tracking_factory->getAvailableAPIS();
    }

    /**
     * @param Order $order
     * @throws PrestaShopDatabaseException
     * @throws PrestaShopException
     */
    public function sendOrderComplete(Order $order)
    {
        if (!empty($this->available_apis)) {
            $order_model = $this->data_model_handler->buildOrderModel($order);
            $order_products = $order->getOrderDetailList();

            if (!empty($order_products)) {
                $user_tracking_data = $this->getUserTrackingData($order->id_customer);
                $traffic_connections = $this->getTrafficConnections($order_model);
                $order_products_list = $this
                    ->data_model_handler
                    ->buildAPIProductModelList($order_products, $order_model->getIdLang())
                    ->getList();
                foreach ($this->available_apis as $api) {
                    $api->setUserTrackingData($user_tracking_data);
                    $api->setPurchaseData($order_model, $order_products_list);
                    $api->setTrafficData($traffic_connections);
                    $api->setCampaignData($traffic_connections);
                    $api->setEventData('ecommerce', 'purchase', '');
                    $api->sendData();
                }
            }
        }
    }

    /**
     * @param $id_customer
     * @return RCPGTagManagerUserTrackingData
     * @throws PrestaShopDatabaseException
     * @throws PrestaShopException
     */
    private function getUserTrackingData($id_customer): RCPGTagManagerUserTrackingData
    {
        return RCPGTagManagerUserTrackingData::getUserTrackingData($id_customer);
    }

    /**
     * @param OrderModel $order_model
     * @return ConnectionsModel
     * @throws PrestaShopDatabaseException
     */
    private function getTrafficConnections(OrderModel $order_model): ConnectionsModel
    {
        return $this->module
            ->getServiceProvider()
            ->getOrderConnectionsService()
            ->getSourceConnection($order_model)
        ;
    }

    /**
     * @param Order $order
     * @throws PrestaShopDatabaseException
     * @throws PrestaShopException
     */
    public function sendOrderRefund(Order $order)
    {
        $order_model = $this->data_model_handler->buildOrderModel($order);
        $order_products = $order->getOrderDetailList();

        if (!empty($order_products)) {
            $user_tracking_data = $this->getUserTrackingData($order->id_customer);

            $order_products_list = $this
                ->data_model_handler
                ->buildOrderProductModelList($order_products, $order_model->getIdLang())
                ->getList()
            ;
            foreach ($this->available_apis as $api) {
                $api->setUserTrackingData($user_tracking_data);
                $api->setRefundData($order_model, $order_products_list);
                $api->setEventData('ecommerce', 'refund', '');
                $api->sendData();
            }
        }
    }

    /**
     * @param Order $order
     * @throws PrestaShopDatabaseException
     * @throws PrestaShopException
     */
    public function sendOrderAborted(Order $order)
    {
        $user_tracking_data = $this->getUserTrackingData($order->id_customer);
        $label = 'order_id_'.$order->id;

        foreach ($this->available_apis as $api) {
            $api->setUserTrackingData($user_tracking_data);
            $api->setEventData('exception', 'order_aborted_by_ad_block', $label);
            $api->sendData();
        }
    }
}
