<?php
/**
 * NOTICE OF LICENSE
 *
 * This source file is subject to a trade license awarded by
 * Garamo Online S.L.
 *
 * Any use, public communication, reproduction, modification or
 * distribution of this source file without the written consent of
 * Garamo Online S.L. It Is prohibited.
 *
 * @author    ReactionCode <info@reactioncode.com>
 * @copyright Garamo Online S.L 2015-2022.
 * @license   Commercial license
 */

use ReactionCode\RCPGTagManager\PSModule\ModuleServiceFactoryInterface;
use ReactionCode\RCPGTagManager\PSModule\RCTrackingModuleInterface;

class RcpgtagmanagerDataRequestModuleFrontController extends ModuleFrontController
{
    /**
     * @var array
     */
    private $request;

    /**
     * @var ModuleServiceFactoryInterface
     */
    private $service_provider;

    /**
     * @var array
     */
    private $response;

    public function postProcess()
    {
        try {
            $this->handleRequest();
        } catch (Exception $exception) {
            http_response_code(400);
            die($exception->getMessage());
        }
    }

    /**
     * @throws PrestaShopDatabaseException
     * @throws PrestaShopException
     * @throws Exception
     */
    private function handleRequest()
    {
        if ($this->module instanceof RCTrackingModuleInterface) {
            $this->service_provider = $this->module->getServiceProvider();
            if ($this->isValidRequest()) {
                $this->request = $this->getRequest();
                $this->request['data'] = $this->parseRequestDataParameter($this->request['data']);
                $this->processAction();
            }
        }
        throw new Exception('invalid request');
    }

    /**
     * @return bool
     */
    private function isValidRequest()
    {
        $request = $this->getRequest();

        return !empty($request['token']) &&
            $this->isValidToken($request['token']) &&
            !empty($request['data']) &&
            !empty($this->parseRequestDataParameter($request['data'])) &&
            !empty($request['action']) &&
            method_exists($this, $request['action']);
    }

    private function getRequest()
    {
        return [
            'token' => Tools::getValue('token', null),
            'module' => Tools::getValue('module', null),
            'action' => Tools::getValue('action', null),
            'data' => Tools::getValue('data', null)
        ];
    }

    /**
     * @param string $request_token
     * @return bool
     */
    private function isValidToken(string $request_token)
    {
        $token = $this->service_provider->getTokenService();
        return $token->getPublicToken() === $request_token;
    }

    protected function parseRequestDataParameter(string $request_data)
    {
        return json_decode(rawurldecode($request_data), true);
    }

    /**
     * @throws PrestaShopDatabaseException
     * @throws PrestaShopException
     */
    private function processAction()
    {
        switch ($this->request['action']) {
            case 'getProductsDetail':
                $this->getProductsDetail();
                break;
            case 'setOrderTrackingStatus':
                $this->setOrderTrackingStatus();
                break;
            case 'handleAdBlockOrderAborted':
                $this->handleAdBlockOrderAborted();
                break;
        }
        $this->sendResponse();
    }

    private function getProductsDetail()
    {
        $data_model_service = $this->service_provider->getDataModelService();
        $this->response = $data_model_service->buildDetailProductModelList($this->request['data'])->getPresentedList();
    }

    /**
     * @throws PrestaShopDatabaseException
     * @throws PrestaShopException
     */
    private function setOrderTrackingStatus()
    {
        $order = new Order($this->request['data']['id']);
        RCPGTagManagerOrderTrackingStatus::setOrderStatus(
            $order,
            RCPGTagManagerOrderTrackingStatus::CUSTOMER_BROWSER
        );
    }

    /**
     * @throws PrestaShopException
     * @throws PrestaShopDatabaseException
     */
    private function handleAdBlockOrderAborted()
    {
        $order = new Order($this->request['data']['id']);
        $api_service = $this->service_provider->getAPIService();
        $api_service->sendOrderComplete($order);
        RCPGTagManagerOrderTrackingStatus::setOrderStatus(
            $order,
            RCPGTagManagerOrderTrackingStatus::SERVER_AD_BLOCK
        );
        $this->response = [];
    }

    private function sendResponse()
    {
        die(json_encode($this->response));
    }
}
